package com.allatori.custom;

/**
 * Custom string encryption/decryption sample.
 */
public class EncryptDecryptString {
    /*
    * Pair of custom string encryption and decryption methods.
    */

    /**
    * Sample encryption method. It just reverses the string ("abc" to "cba"). For test purposes only!
    *
    * This method will get all string literals from your application during the obfuscation process (if string encryption is set to 'maximum'). It returns encrypted version of the string.
    * The method is needed during the obfuscation process only, it is not needed in the runtime of your application.
    */
    public static String encryptString(String s) {
        return new StringBuilder(s).reverse().toString();
    }

    /**
    * Sample decryption method. It just reverses the string ("cba" to "abc"). For test purposes only!
    *
    * The method is needed in the runtime and should be placed in some class of your application.
    */
    public static String decryptString(String s) {
        return new StringBuilder(s).reverse().toString();
    }


    /*
    * Another way of using custom string encryption.
    * Encryption method is provided with class name and method name of the string's location. Decryption method gets this information in the runtime.
    * This way, the same string used in different classes/methods will be encrypted differently.
    *
    * Pair of custom string encryption and decryption methods tied to string's location.
    */

    /**
    * Sample encryption method. It adds class and method names to the string and then reverses the whole string ("abc" to "cba"). For test purposes only!
    *
    * This method will get all string literals from your application during the obfuscation process (if string encryption is set to 'maximum'). It returns encrypted version of the string.
    * The method is needed during the obfuscation process only, it is not needed in the runtime of your application.
    */
    public static String encryptStringLocationTied(String s, String className, String methodName) {
        return new StringBuilder(className + s + methodName).reverse().toString();
    }

    /**
    * Sample decryption method. It reverses the string ("cba" to "abc") and removes added class and method names. For test purposes only!
    *
    * The method is needed in the runtime and should be placed in some class of your application.
    */
    public static String decryptStringLocationTied(String s) {
        // getting string's location class and method names in the runtime
        StackTraceElement element = new Exception().getStackTrace()[1];
        String className = element.getClassName();
        String methodName = element.getMethodName();
        // decrypting
        String reversedString = new StringBuilder(s).reverse().toString();
        return reversedString.substring(className.length(), reversedString.length() - methodName.length());
    }


    /*
    * Using custom string encryption for internationalization
    */

    /**
    * As encryption method gets all string literals from your application during the obfuscation process (if string encryption is set to 'maximum'),
    * you can make a list of all strings used in your application for internationalization.
    */
    public static String encryptStringI18N(String s) {
        // log string for internationalization
        return s;
    }

    /**
    * This method will be called for all string literals in your application - "abc" would become decryptStringI18N("abc"),
    * so you can replace strings with internationalized version in the runtime.
    */
    public static String decryptStringI18N(String s) {
        // read internationalized version of s from file and return it

        // test, replacing "hello!" with "aloha!"
        if ("hello!".equals(s)) return "aloha!";
        return s;
    }


    /**
    * Run test.
    */
    public static void main(String[] args) {
        String s = "Hello, World!";
        System.out.println("Option #1\nOriginal string:\t" + s);
        String encrypted = EncryptDecryptString.encryptString(s);
        // encrypted string could contain non-printable characters, you might want to comment out the line below
        System.out.println("Encrypted string:\t" + encrypted);
        String decrypted = EncryptDecryptString.decryptString(encrypted);
        System.out.println("Decrypted string:\t" + decrypted);

        s = "Hello, World!";
        System.out.println("\nOption #2\nOriginal string (location tied):\t" + s);
        String encryptedLocationTied = EncryptDecryptString.encryptStringLocationTied(s, "com.allatori.custom.EncryptDecryptString", "main");
        // encrypted string could contain non-printable characters, you might want to comment out the line below
        System.out.println("Encrypted string (location tied):\t" + encryptedLocationTied);
        String decryptedLocationTied = EncryptDecryptString.decryptStringLocationTied(encryptedLocationTied);
        System.out.println("Decrypted string (location tied):\t" + decryptedLocationTied);

        s = "hello!";
        System.out.println("\nOption #3\nOriginal string (i18n):\t\t" + s);
        String encryptedI18N= EncryptDecryptString.encryptStringI18N(s);
        System.out.println("Encrypted string (i18n):\t" + encryptedI18N);
        String decryptedI18N= EncryptDecryptString.decryptStringI18N(encryptedI18N);
        System.out.println("Decrypted string (i18n):\t" + decryptedI18N);
    }
}
